<?php
/**
 * Library for generating Pimcore object class
 * Abstract model, must be implemented for generating modules. Ex: Manager, API, etc.
 *
 * @author Suhar Prasetyo suhar.tyo@gmail.com
 * @copyright 2017 Dreamcube Id
 * @version 1.0 In development, very unstable
 * @package Website
 * @subpackage Website\Generator
 * @category Library For pimcore and use standard coding by Backend team in Dreamcube Id 
 */

namespace AppHelper\Generator;


abstract class AbstractModel
{

	/**
	 * @var Website\Generator\ObjectData
	 */
	protected $_object;

	/**
	 * Script that will be generated
	 * @var String
	 */
	protected $_script;

	/**
	 * True if you want to overwrite existing file, False if you dont
	 * 
	 * @var Boolean
	 */
	protected $_overwrite;


	public function __construct($overwrite = false)
	{
		$this->_overwrite = $overwrite;
	}

	/**
	 * Start script that will be generated
	 * 
	 * @param  \stdClass $params Nullable. Set if you dont should pass parameter
	 * @return void
	 */
	public abstract function start($params = null);

	/**
	 * Finish script that will be generated
	 * 
	 * @param  \stdClass $params Nullable. Set if you dont should pass parameter
	 * @return void
	 */
	public abstract function finish($params = null);

	/**
	 * Body script that will be generated
	 * 
	 * @param  \stdClass $params Nullable. Set if you dont should pass parameter 
	 * @return [type]         [description]
	 */
	public abstract function body($params = null);

	/**
	 * Generate script
	 * 
	 * @param  \stdClass $params Nullable. Set if you dont should pass parameter
	 * @return Boolean
	 */
	public abstract function generate($params = null);

	/**
	 * Is process detail data
	 * 
	 * @return Boolean
	 */
	public abstract function isProcessData();

	/**
	 * This method should be call before you do anything
	 * 
	 * @param ObjectData $object
	 */
	public function setObjectData(ObjectData $object)
	{
		$this->_object = $object;
	}

	/**
	 * Set script, override existing script if needed
	 * 
	 * @param String $script
	 */
	public function setScript($script)
	{
		$this->_script = $script;
	}

	/**
	 * Append script, append existing script if needed
	 * 
	 * @param String $script
	 */
	public function appendScript($script)
	{
		$this->_script .= $script;
	}

	/**
	 * Get script
	 * 
	 * @return String $_script
	 */
	public function getScript()
	{
		return $this->_script;
	}

	/**
	 * Get ObjectData
	 * 
	 * @return ObjectData $_object
	 */
	public function getObjectData()
	{
		return $this->_object;
	}

	/**
	 * Add data to model
	 * 
	 * @param \Model\DataObject\ClassDefinition\Data $data  
	 * @return void
	 */
	public function addData(\Pimcore\Model\DataObject\ClassDefinition\Data $data)
	{
		$this->autoCreateObjectData();
		$this->_object->addData($data);
	}

	/**
	 * Remove data from model by data name
	 * 
	 * @param  String $name 
	 * @return void       
	 */
	public function removeData($name)
	{
		$this->autoCreateObjectData();
		$this->_object->removeData($data);
	}

	/**
	 * Clear all properties value
	 * 
	 * @return void       
	 */
	public function clear()
	{
		$this->_script = '';
		$this->_object = null;
	}

	/**
	 * Generate tab
	 * 
	 * @param  integer $tab 
	 * @return String       
	 */
	public function tab($tab = 1)
	{

		$tabs = "";
		for ($i=0; $i < $tab; $i++) { 
			$tabs .= "\t";
		}

		return $tabs;
	}

	/**
	 * Generate new line
	 * @param  integer $line
	 * @return String       
	 */
	public function line($line = 1)
	{

		$lines = "";
		for ($i=0; $i < $line; $i++) { 
			$lines .= PHP_EOL;
		}
		
		return $lines;
	}

	/**
	 * Auto create new instance of ObjectData if you never call setObjectData()
	 * 
	 * @return void
	 */
	private function autoCreateObjectData()
	{
		if (!$this->_object) $this->_object = new ObjectData(0,'','');
	}

	
}