<?php
/**
 * Library for generating Pimcore object class
 * 
 * Class that handled generating Pimcore object class
 * Example : 
 * 	$manager = new AppHelper\Generator\Adapter(new \AppHelper\Generator\Lib\ManagerModel());
 * 	$manager->generate();
 * 	$api = new AppHelper\Generator\Adapter(new \AppHelper\Generator\Lib\ApiModel());
 * 	$api->generate();
 *
 * If you want to to generate by class id 
 * 	Call this method $api->generate(1);
 * 	Call this method $manager->generate(1);
 * 
 * @author Suhar Prasetyo suhar.tyo@gmail.com
 * @copyright 2017 Dreamcube Id
 * @version 1.0 In development, very unstable
 * @package AppBundle\Utils
 * @subpackage AppBundle\Utils\Generator
 * @category Library For pimcore and use standard coding by Backend team in Dreamcube Id
 */

namespace AppHelper\Generator;


class Adapter
{

	/**
	 * @var array[string] Should contain error messages
	 */
	protected $_error_messages = [];

	/**
	 * Model thath will be generated
	 * @var AbstractModel 
	 */
	protected $_model;


	function __construct(AbstractModel $model)
	{
		$this->_model = $model;
	}

	/**
	 * Function for generating manager and API all object classes
	 * 
	 * @param  int $id Allow null. id class, 
	 *                 if empty then generating all object class, else generating single object
	 * @return boolean	true if generate success, false if generate fail
	 */
	public function generate($id = null)
	{

		if ($id) return $this->generateSingle($id);

		$query = "SELECT id FROM classes";
		$classes = \AppHelper\Mysql::makeQuery($query);

		if (!count($classes)) {
			$this->addErrorMessage('Class not found');
			return false;
		}

		$result = false;
		foreach ($classes as $key_class => $class) {
			$single = $this->generateSingle($class['id']);
			if ($single) $result = true;
		}
		
		return $result;

	}

	/**
	 * Function for generating manager and API single object classes 
	 * 
	 * @param  int $id not null. id class  
	 * @return boolean	true if generate success, false if generate fail
	 */
	public function generateSingle($id)
	{

		$this->_model->clear();

		$query = "SELECT id, name FROM classes WHERE id = ?";
		$classes = \AppHelper\Mysql::makeQuery($query, [$id]);

		if (!count($classes)) {
			$this->addErrorMessage("Class ID {$id} not found");
			return false;
		}

		$object_data = new ObjectData($classes[0]['id'], $classes[0]['name'], $classes[0]['description']);
		$this->_model->setObjectData($object_data);

		$class = include(PIMCORE_CLASS_DIRECTORY ."/definition_{$classes[0]['name']}.php");

        $this->_model->start();
        if ($this->_model->isProcessData()) $this->processChildData($class);
        $this->_model->body();
        $this->_model->finish();

        $this->_model->generate();

		return true;

	}

	/**
	 * Add error message
	 * @param string $message
	 */
	public function addErrorMessage($message)
	{
		array_push($this->_error_messages, $message);
	}

	/**
	 * Get error messages from proccess generating object class/classes
	 * @return array[string] List of error messages
	 */
	public function getErrorMessages()
	{
		return $this->_error_messages;
	}
	
	/**
	 * Processing data field
	 *
	 * This method will call methods from $_model recursively by parent
	 * 
	 * @param $parent 
	 * @return void                                     
	 */
	protected function processChildData($parent)
	{

		foreach ($parent->getFieldDefinitions() as $key => $object) {
			if ($object instanceof \Pimcore\Model\DataObject\ClassDefinition\Layout\Panel) {
				$this->processChildData($object);
			} elseif ($object instanceof \Pimcore\Model\DataObject\ClassDefinition\Layout\Tabpanel) {
				$this->processChildData($object);
			} else {
				$this->_model->addData($object);
			}
		}

	}

}