<?php

namespace OauthBundle\EventSubscriber;

use Symfony\Component\HttpKernel\Event\FilterResponseEvent;
use Symfony\Component\HttpKernel\Event\FilterControllerEvent;
use Doctrine\Common\Util\ClassUtils;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Lcobucci\JWT\Parser;
use Lcobucci\JWT\ValidationData;


/**
* 
*/
class OauthSubscriber implements EventSubscriberInterface
{

	protected $_reader;

	function __construct($reader)
	{
		$this->_reader = $reader;
	}

	public static function getSubscribedEvents()
    {
        return array(
            KernelEvents::CONTROLLER => 'onKernelController',
	        KernelEvents::RESPONSE => 'onKernelResponse',
        );
    }

	public function onKernelController(FilterControllerEvent $event)
	{

		$controller = $event->getController();

        if (!is_array($controller)) {
            return;
        }

        list($controller_object, $method_name) = $controller;

        $this->handleAuthToken($event, $controller_object, $method_name);
        $this->handleApiKey($event, $controller_object, $method_name);
        
	}

	public function onKernelResponse(FilterResponseEvent $event)
	{
		
	}

    public function handleAuthToken(FilterControllerEvent $event, $controller_object, $method_name)
    {
        $session = $event->getRequest()->getSession();
        $auth_token = $session->get('auth_token');
        // $auth_token = $event->getRequest()->headers->get('auth_token');
        $annotation = 'OauthBundle\Annotation\Oauth';

        // Get class annotation
        // Using ClassUtils::getClass in case the controller is an proxy
        $class_annotation = $this->_reader->getClassAnnotation(
            new \ReflectionClass(ClassUtils::getClass($controller_object)), $annotation
        );

        // Get method annotation
        $controller_reflection_object = new \ReflectionObject($controller_object);
        $reflectionMethod = $controller_reflection_object->getMethod($method_name);
        $method_annotation = $this->_reader->getMethodAnnotation($reflectionMethod,$annotation);

        if ($method_annotation || $class_annotation) {

            if ($auth_token) {
                $token = (new Parser())->parse((string) $auth_token);
                $data = new ValidationData();
                $data->setIssuer($event->getRequest()->getSchemeAndHttpHost());
                $data->setAudience($event->getRequest()->getSchemeAndHttpHost());
                $is_validate = $token->validate($data);

                if ($is_validate) {
                    $event->getRequest()->request->set('oauth_user_id', $token->getClaim('UserId'));
                    $event->getRequest()->request->set('oauth_name', $token->getClaim('Name'));
                    $event->getRequest()->request->set('oauth_email', $token->getClaim('Email'));
                    $event->getRequest()->request->set('oauth_role', $token->getClaim('Role'));
                    $event->getRequest()->request->set('oauth_scope', $token->getClaim('Scope'));
                }
            }

            if ($method_annotation && $method_annotation->isRequired()) {
                if (!$is_validate) { 
                    throw new AccessDeniedHttpException('Access token is required');
                }
            } elseif (!$method_annotation && $class_annotation && $class_annotation->isRequired()) {
                if (!$is_validate) {
                    throw new AccessDeniedHttpException('Access token is required');
                }
            }
        }
    }

    public function handleApiKey(FilterControllerEvent $event, $controller_object, $method_name)
    {
        $apikey = $event->getRequest()->headers->get('apikey');
        $annotation = 'OauthBundle\Annotation\ApiKey';

        // Get class annotation
        // Using ClassUtils::getClass in case the controller is an proxy
        $class_annotation = $this->_reader->getClassAnnotation(
            new \ReflectionClass(ClassUtils::getClass($controller_object)), $annotation
        );

        // Get method annotation
        $controller_reflection_object = new \ReflectionObject($controller_object);
        $reflectionMethod = $controller_reflection_object->getMethod($method_name);
        $method_annotation = $this->_reader->getMethodAnnotation($reflectionMethod,$annotation);

        if ($method_annotation || $class_annotation) {
            $params = new \stdClass;
            $params->ApiKey = $apikey;
            $params->Domain = $_SERVER['SERVER_NAME'];

            $client = \AppModel\OauthClient::checkOauthClient($params);

            if ($method_annotation && $method_annotation->isRequired()) {       
                $scope = $method_annotation->getScope();
                $array_scope = explode(",", $scope);
                if (!$client->Status) {
                    throw new AccessDeniedHttpException('API Key is required');
                } else {
                    if (!empty(array_intersect($array_scope, $client->Data->Scope))) {
                        $event->getRequest()->request->set('oauth_api_user', $client->Data);
                    } else {
                        throw new AccessDeniedHttpException('Cannot access this API');
                    }
                } 
            } elseif (!$method_annotation && $class_annotation && $class_annotation->isRequired()) {
                $scope = $class_annotation->getScope();
                $array_scope = explode(",", $scope);
                if (!$client->Status) {
                    throw new AccessDeniedHttpException('API Key is required');
                } else {
                    if (!empty(array_intersect($array_scope, $client->Data->Scope))) {
                        $event->getRequest()->request->set('oauth_api_user', $client->Data);
                    } else {
                        throw new AccessDeniedHttpException('Cannot access this API');
                    }
                } 
            }
        }
    }

}